#!/bin/bash

# MacAgent Pro Installer Script
# Handles macOS security and quarantine issues automatically

echo "🚀 Installing MacAgent Pro..."

# Get the directory where this script is located
SCRIPT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
ZIP_FILE="$SCRIPT_DIR/MacAgent-Pro-v1.0.2-Signed.zip"
TEMP_DIR="/tmp/macagent-install"
APP_NAME="MacAgent Pro.app"
INSTALL_DIR="/Applications/$APP_NAME"

# Check if zip file exists
if [ ! -f "$ZIP_FILE" ]; then
    echo "❌ Error: MacAgent-Pro-v1.0.2-Signed.zip not found"
    echo "Please download the complete package from macagent.pro"
    exit 1
fi

# Create temp directory
rm -rf "$TEMP_DIR"
mkdir -p "$TEMP_DIR"

# Extract the app
echo "📦 Extracting MacAgent Pro..."
cd "$TEMP_DIR"
unzip -q "$ZIP_FILE"

if [ ! -d "$APP_NAME" ]; then
    echo "❌ Error: MacAgent Pro.app not found in zip file"
    exit 1
fi

# Remove any existing installation
if [ -d "$INSTALL_DIR" ]; then
    echo "🗑️  Removing previous installation..."
    sudo rm -rf "$INSTALL_DIR"
fi

# Copy to Applications
echo "📋 Installing to Applications folder..."
sudo cp -R "$APP_NAME" "/Applications/"

# Fix permissions
echo "🔧 Setting correct permissions..."
sudo chmod -R 755 "$INSTALL_DIR"
sudo chmod +x "$INSTALL_DIR/Contents/MacOS/MacAgent Pro"

# Remove quarantine attribute
echo "🔓 Removing macOS quarantine restrictions..."
sudo xattr -rd com.apple.quarantine "$INSTALL_DIR" 2>/dev/null || true

# Sign the app (ad-hoc)
echo "🔐 Applying code signature..."
sudo codesign --force --deep --sign - "$INSTALL_DIR" 2>/dev/null || echo "⚠️  Code signing skipped (optional)"

# Create launch script for easier access
cat > "/tmp/launch-macagent.command" << 'EOF'
#!/bin/bash
open "/Applications/MacAgent Pro.app"
EOF

chmod +x "/tmp/launch-macagent.command"
cp "/tmp/launch-macagent.command" "$HOME/Desktop/Launch MacAgent Pro.command"

# Cleanup
rm -rf "$TEMP_DIR"

echo ""
echo "✅ MacAgent Pro installed successfully!"
echo ""
echo "🚀 To launch:"
echo "   • Double-click 'Launch MacAgent Pro' on your Desktop"
echo "   • Or open Applications and double-click MacAgent Pro"
echo "   • Look for 🤖 in your menubar when running"
echo ""
echo "📧 If you still get security warnings:"
echo "   1. Right-click MacAgent Pro.app → Open"
echo "   2. Click 'Open' when prompted"
echo "   3. App will be permanently allowed"
echo ""
echo "🎯 MacAgent Pro is now ready to use!"